const express = require('express');
const cors = require('cors');
const sqlite3 = require('sqlite3');
const { open } = require('sqlite');

const app = express();
app.use(cors());
app.use(express.json());

// SQLite DB setup
let db;

// Ensure DB and tables are set up on server start
(async () => {
  db = await open({
    filename: './site.db',
    driver: sqlite3.Database
  });
  await db.exec(`CREATE TABLE IF NOT EXISTS settings (
    key TEXT PRIMARY KEY,
    value TEXT
  )`);
  await db.exec(`CREATE TABLE IF NOT EXISTS pages (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    name TEXT NOT NULL,
    link TEXT,
    content TEXT
  )`);
  await db.exec(`CREATE TABLE IF NOT EXISTS videos (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    name TEXT NOT NULL,
    type TEXT NOT NULL, -- 'local', 'youtube', 'live', etc.
    link TEXT NOT NULL
  )`);
  // Users table for authentication
  await db.exec(`CREATE TABLE IF NOT EXISTS users (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    username TEXT UNIQUE NOT NULL,
    password TEXT NOT NULL,
    is_admin INTEGER DEFAULT 0
  )`);

  // Optionally, insert default menu pages if table is empty
  const pageCount = (await db.get('SELECT COUNT(*) as count FROM pages')).count;
  if (pageCount === 0) {
    await db.run("INSERT INTO pages (name, link, content) VALUES ('Home', '/', 'Welcome to alo@laba!')");
    await db.run("INSERT INTO pages (name, link, content) VALUES ('About', '/about', 'About us page')");
    await db.run("INSERT INTO pages (name, link, content) VALUES ('Contact', '/contact', 'Contact info page')");
  }



const bcrypt = require('bcryptjs');
const jwt = require('jsonwebtoken');

const JWT_SECRET = process.env.JWT_SECRET || 'alo@laba_secret';

app.post('/api/register', async (req, res) => {
  const { username, password, is_admin } = req.body;
  if (!username || !password) return res.status(400).json({ error: 'Username and password required' });
  const hash = await bcrypt.hash(password, 10);
  try {
    await db.run('INSERT INTO users (username, password, is_admin) VALUES (?, ?, ?)', [username, hash, is_admin ? 1 : 0]);
    res.json({ success: true });
  } catch (e) {
    res.status(400).json({ error: 'User already exists' });
  }
});

// Login
app.post('/api/login', async (req, res) => {
  const { username, password } = req.body;
  if (!username || !password) return res.status(400).json({ error: 'Username and password required' });
  const user = await db.get('SELECT * FROM users WHERE username = ?', [username]);
  if (!user) return res.status(401).json({ error: 'Invalid credentials' });
  const valid = await bcrypt.compare(password, user.password);
  if (!valid) return res.status(401).json({ error: 'Invalid credentials' });
  const token = jwt.sign({ id: user.id, username: user.username, is_admin: !!user.is_admin }, JWT_SECRET, { expiresIn: '1d' });
  res.json({ token, is_admin: !!user.is_admin });
});

// Middleware to check admin
function requireAdmin(req, res, next) {
  const auth = req.headers.authorization;
  if (!auth) return res.status(401).json({ error: 'No token' });
  try {
    const decoded = jwt.verify(auth.split(' ')[1], JWT_SECRET);
    if (!decoded.is_admin) return res.status(403).json({ error: 'Admin only' });
    req.user = decoded;
    next();
  } catch {
    res.status(401).json({ error: 'Invalid token' });
  }
}

// Example admin-only route
app.get('/api/admin/check', requireAdmin, (req, res) => {
  res.json({ admin: true, user: req.user });
});
})();

// Serve React frontend build for all non-API routes
const path = require('path');
app.use(express.static(path.join(__dirname, '../frontend/build')));
app.get(/^((?!\/api).)*$/, (req, res) => {
  res.sendFile(path.join(__dirname, '../frontend/build/index.html'));
});
// Example API routes
app.get('/api/settings', async (req, res) => {
  const settings = await db.all('SELECT * FROM settings');
  res.json(settings);
});

app.get('/api/pages', async (req, res) => {
  const pages = await db.all('SELECT * FROM pages');
  res.json(pages);
});

app.post('/api/settings', async (req, res) => {
  const { key, value } = req.body;
  await db.run('INSERT OR REPLACE INTO settings (key, value) VALUES (?, ?)', [key, value]);
  res.json({ success: true });
});

const https = require('https');
const fs = require('fs');
const PORT = process.env.PORT || 5000;
const options = {
  key: fs.readFileSync('./server.key'),
  cert: fs.readFileSync('./server.cert')
};
https.createServer(options, app).listen(PORT, () => {
  console.log(`Backend running in production mode on https://localhost:${PORT}`);
});
